#!/usr/bin/env python3
"""
XOR encrypts all .m4a files in the Drums directory in place.
"""

import os
import sys
import argparse
from pathlib import Path

# Default encryption key (you should change this to a secure key)
DEFAULT_KEY = b'iRealProDrums2024!'

def xor_encrypt_file(file_path, key):
    """
    XOR encrypt/decrypt a file in place.
    Since XOR is symmetric, the same operation encrypts and decrypts.
    """
    try:
        # Read the file
        with open(file_path, 'rb') as f:
            data = f.read()
        
        # XOR encrypt the data
        encrypted_data = bytearray()
        for i, byte in enumerate(data):
            encrypted_data.append(byte ^ key[i % len(key)])
        
        # Write back to the same file
        with open(file_path, 'wb') as f:
            f.write(encrypted_data)
        
        return True
    except Exception as e:
        print(f"Error processing {file_path}: {e}")
        return False

def encrypt_drums_directory(drums_dir, key=DEFAULT_KEY, dry_run=False):
    """
    Recursively encrypt all .m4a files in drums_dir.
    Only encrypt files that are not already encrypted (i.e., have the 'ftyp' header).
    """
    drums_path = Path(drums_dir)
    
    if not drums_path.exists():
        print(f"Error: Drums directory '{drums_dir}' does not exist")
        return False
    
    # Find all .m4a files
    m4a_files = list(drums_path.rglob("*.m4a"))
    
    if not m4a_files:
        print(f"No .m4a files found in {drums_dir}")
        return True
    
    print(f"Found {len(m4a_files)} .m4a files to process")
    
    if dry_run:
        print("DRY RUN - No files will be modified")
        for file_path in m4a_files:
            if has_ftyp_header(file_path):
                print(f"  Would encrypt: {file_path}")
            else:
                print(f"  Skipping (already encrypted): {file_path}")
        return True
    
    # Process each file
    success_count = 0
    skipped_count = 0
    to_encrypt = [f for f in m4a_files if has_ftyp_header(f)]
    for file_path in m4a_files:
        if has_ftyp_header(file_path):
            print(f"Encrypting: {file_path}")
            if xor_encrypt_file(file_path, key):
                success_count += 1
            else:
                print(f"Failed to encrypt: {file_path}")
        else:
            print(f"Skipping (already encrypted): {file_path}")
            skipped_count += 1
            success_count += 1
    
    print(f"Successfully encrypted {success_count - skipped_count}/{len(m4a_files)} files (skipped {skipped_count} already encrypted)")
    return True

def decrypt_drums_directory(drums_dir, key=DEFAULT_KEY, dry_run=False):
    """
    Recursively decrypt all .m4a files in drums_dir.
    Only decrypt files that are encrypted (i.e., missing the 'ftyp' header).
    """
    drums_path = Path(drums_dir)
    
    if not drums_path.exists():
        print(f"Error: Drums directory '{drums_dir}' does not exist")
        return False
    
    # Find all .m4a files
    m4a_files = list(drums_path.rglob("*.m4a"))
    
    if not m4a_files:
        print(f"No .m4a files found in {drums_dir}")
        return True
    
    print(f"Found {len(m4a_files)} .m4a files to process for decryption")
    
    to_decrypt = [f for f in m4a_files if not has_ftyp_header(f)]
    
    if dry_run:
        print("DRY RUN - No files will be modified")
        for file_path in m4a_files:
            if not has_ftyp_header(file_path):
                print(f"  Would decrypt: {file_path}")
            else:
                print(f"  Skipping (not encrypted): {file_path}")
        return True
    
    # Process each file
    success_count = 0
    skipped_count = 0
    for file_path in m4a_files:
        if not has_ftyp_header(file_path):
            print(f"Decrypting: {file_path}")
            if xor_encrypt_file(file_path, key):  # XOR is symmetric
                success_count += 1
            else:
                print(f"Failed to decrypt: {file_path}")
        else:
            print(f"Skipping (not encrypted): {file_path}")
            success_count += 1
            skipped_count += 1
    
    print(f"Successfully decrypted {success_count}/{len(m4a_files)} files (skipped {skipped_count} not encrypted)")
    return True

def has_ftyp_header(file_path):
    """
    Check if the file contains 'ftyp' in the first 32 bytes (typical for .m4a files).
    We use this to determine if the file is encrypted
    Returns True if found, False otherwise.
    """
    try:
        with open(file_path, 'rb') as f:
            header = f.read(32)
            return b'ftyp' in header
    except Exception as e:
        print(f"Error reading {file_path}: {e}")
        return False

def check_encrypted_files(drums_dir):
    """
    Check all .m4a files in the directory for the 'ftyp' header.
    """
    drums_path = Path(drums_dir)
    if not drums_path.exists():
        print(f"Error: Drums directory '{drums_dir}' does not exist")
        return False
    m4a_files = list(drums_path.rglob("*.m4a"))
    if not m4a_files:
        print(f"No .m4a files found in {drums_dir}")
        return True
    print(f"Checking {len(m4a_files)} .m4a files for encryption status:")
    for file_path in m4a_files:
        if has_ftyp_header(file_path):
            print(f"  {file_path}: NOT encrypted (ftyp header found)")
        else:
            print(f"  {file_path}: ENCRYPTED or not a valid m4a (ftyp header missing)")
    return True

def main():
    parser = argparse.ArgumentParser(description='Encrypt or decrypt drum sample files with XOR encryption')
    parser.add_argument('--drums-dir', default='.',
                       help='Path to the Drums directory (default: .)')
    parser.add_argument('--key', default=DEFAULT_KEY.decode('utf-8'),
                       help='Encryption key (default: iRealProDrums2024!)')
    parser.add_argument('--dry-run', action='store_true',
                       help='Show what would be encrypted/decrypted without actually doing it')
    parser.add_argument('--verbose', '-v', action='store_true',
                       help='Verbose output')
    parser.add_argument('--check-encrypted', action='store_true',
                       help='Check if .m4a files are already encrypted (missing ftyp header)')
    parser.add_argument('--decrypt', action='store_true',
                       help='Decrypt .m4a files that are encrypted (missing ftyp header)')
    
    args = parser.parse_args()
    
    if args.verbose:
        print(f"Drums directory: {args.drums_dir}")
        print(f"Encryption key: {args.key}")
        print(f"Dry run: {args.dry_run}")
        print(f"Check encrypted: {args.check_encrypted}")
        print(f"Decrypt: {args.decrypt}")
    
    if args.check_encrypted:
        # Only check encryption status, then exit
        result = check_encrypted_files(args.drums_dir)
        sys.exit(0 if result else 1)
    
    # Convert key to bytes
    key = args.key.encode('utf-8')
    
    if args.decrypt:
        # Decrypt files
        success = decrypt_drums_directory(args.drums_dir, key, args.dry_run)
        if success:
            print("Decryption completed successfully")
            sys.exit(0)
        else:
            print("Decryption failed")
            sys.exit(1)
    else:
        # Encrypt files
        success = encrypt_drums_directory(args.drums_dir, key, args.dry_run)
        if success:
            print("Encryption completed successfully")
            sys.exit(0)
        else:
            print("Encryption failed")
            sys.exit(1)

if __name__ == "__main__":
    main() 
